# Singly Linked List

## 1. Singly Linked List Overview

A **singly linked list** is a linear data structure made of nodes, where each node stores:

- a data element, and
- a reference to the **next** node in the sequence.

Unlike arrays, a linked list:

- supports efficient insertion/removal at the **head** (front),
- allows traversal in **one direction only** (from head to tail).

This implementation is **generic** (`SinglyLinkedList<T>`), so it can store any object type.

---

## 2. Files in This Project

- **`Position.java`**  
  A minimal generic interface representing a position in a container:

  - `T getElement()`: returns the element stored at this position.

- **`SNode.java`**  
  Node implementation for the singly linked list:

  - Stores `element` of type `T` and a reference to `next` node.
  - Implements `Position<T>`.

- **`EmptyListException.java`**  
  Custom checked exception thrown when attempting operations that are invalid on an empty list (e.g., removing or accessing head/tail when the list is empty).

- **`SList.java`**  
  Interface describing the **singly linked list ADT**:

  - `int size()`
  - `boolean isEmpty()`
  - `void insertAtHead(T element)`
  - `void insertAtTail(T element)`
  - `T removeFromHead() throws EmptyListException`
  - `T removeFromTail() throws EmptyListException`
  - `T getHead() throws EmptyListException`
  - `T getTail() throws EmptyListException`

- **`SinglyLinkedList.java`**  
  Concrete implementation of `SList<T>` using `SNode<T>` and maintaining `head`, `tail`, and `size`.

- **`SinglyLinkedListDemo.java`**  
  Simple driver class demonstrating basic list operations (inserting at head/tail, removing, printing).

- **`Stack.java`**  
  Interface for a generic stack ADT:

  - `size`, `isEmpty`, `push`, `pop`, `top`.

- **`StackException.java`**  
  Custom exception for invalid stack operations (e.g., `pop` on an empty stack).

- **`SLLBasedStack.java`**  
  Stack implementation **backed by** `SinglyLinkedList<T>`, illustrating how the list can serve as a building block for other data structures.

---

## 3. `SinglyLinkedList.java` – Implementation Details

### 3.1 Internal Representation

```java
private SNode<T> head, tail;
private int size;
```

- **`head`**: reference to the first node (or `null` if list is empty).
- **`tail`**: reference to the last node (or `null` if list is empty).
- **`size`**: number of elements in the list.

Maintaining both `head` and `tail` allows **O(1)** insertion at both ends (except for removal from tail, which requires traversal in a singly linked list).

---

### 3.2 Core Methods and Their Behavior

Below is the intended behavior of the key methods defined in `SList<T>` and implemented in `SinglyLinkedList<T>`.

#### `int size()`

- Returns the current number of elements (`size` field).
- **Time complexity:** `O(1)`.

#### `boolean isEmpty()`

- Returns `true` if `size == 0`, `false` otherwise.
- **Time complexity:** `O(1)`.

#### `void insertAtHead(T element)`

Typical logic:

1. Create a new node `newNode` whose `next` points to the current `head`.
2. Update `head` to point to `newNode`.
3. If the list was empty, also set `tail = newNode`.
4. Increment `size`.

- **Time complexity:** `O(1)`.

#### `void insertAtTail(T element)`

Thanks to the `tail` pointer, we can insert in constant time:

Typical logic:

1. Create a new node `newNode` with `next = null`.
2. If the list is empty, set both `head` and `tail` to `newNode`.
3. Otherwise, set `tail.setNext(newNode)` and then update `tail = newNode`.
4. Increment `size`.

- **Time complexity:** **`O(1)`**, because we do **not** traverse the list; we use the direct `tail` reference.
- This is a major advantage over a singly linked list that only maintains a `head` pointer, where tail insertion would be `O(n)`.

#### `T removeFromHead() throws EmptyListException`

Typical logic:

1. If the list is empty, throw `EmptyListException`.
2. Store the current `head` node in a temporary variable.
3. Set `head = head.getNext()`.
4. If the list becomes empty (i.e., `head == null`), also set `tail = null`.
5. Decrement `size`.
6. Return the element of the removed node.

- **Time complexity:** `O(1)`.

#### `T removeFromTail() throws EmptyListException`

For a singly linked list, removing from tail requires finding the **node just before the tail**:

Typical logic:

1. If the list is empty, throw `EmptyListException`.
2. If `size == 1`, store `head`’s element, set both `head` and `tail` to `null`, set `size = 0`, return the element.
3. Otherwise, traverse from `head` until you reach the node whose `next` is `tail`:

   - This node becomes the new `tail`.

4. Set `tail.setNext(null)`.
5. Decrement `size`.
6. Return the removed element.

- **Time complexity:** `O(n)`, due to the traversal needed to find the node before `tail`.

#### `T getHead() throws EmptyListException`

- If the list is empty, throw `EmptyListException`.
- Otherwise, return `head.getElement()`.
- **Time complexity:** `O(1)`.

#### `T getTail() throws EmptyListException`

- If the list is empty, throw `EmptyListException`.
- Otherwise, return `tail.getElement()`.
- **Time complexity:** `O(1)`.

#### `String toString()`

- Iterates from `head` to `null`, appending each element to a `StringBuilder`.
- Returns a space-separated representation of the list’s contents.
- **Time complexity:** `O(n)`.
